#include "General.h"
#include "ScreenShakeConsoleCommand.h"
#include "DamageableGameObj.h"
#include "CommandLineParser.h"
#include "engine_tt.h"
#include "engine_io.h"
#include "engine_game.h"
#include "ScriptableGameObj.h"
#include "DefenseObjectClass.h"
#include "gmgame.h"
#include "Iterator.h"

static int getClientIdByIdentifier(const char* clientIdentifier)
{
	assert(clientIdentifier);
	const cPlayer* player = Find_Player(atoi(clientIdentifier));
	int result;
	if (player)
		result = player->PlayerId;
	else
		result = -1;
	return result;
}

static bool isClientId(const int id)
{
	return id > 0 && id < 128 && Find_Player(id);
}

class CommandSCREENSHAKE : public ConsoleFunctionClass
{
public:
	const char* Get_Name() 
	{ 
		return "screenshake"; 
	}
	const char* Get_Help() 
	{ 
		return "SCREENSHAKE <ID> <DURATION> - Shakes the screen of a player for a variable DURATION.\n"; 
	}
	void Activate(const char* argumentsString)
	{
		CommandLineParser arguments(argumentsString);
		const char* clientIdentifier = arguments.getString();

		if (!clientIdentifier || clientIdentifier[0] == '\0')
			Console_Output("Please a player ID.\n");
		else
		{
			const int clientId = getClientIdByIdentifier(clientIdentifier);
			if (!isClientId(clientId))
			{
				Console_Output("Please enter a valid player ID.");
			}
			else
			{
				int Duration = arguments.getInt();

				if (Duration != -1) 
				{	
					StringClass Params = StringClass::getFormattedString("%d", Duration);
					Attach_Script_Once(Get_GameObj(clientId), "ScreenShake_Script", Params);
				}
				else
				{
					Console_Output("Please enter a (valid) screen shake duration\n");
				}
			}
		}
	}
};

void Shake_Camera_Player(int PlayerID, Vector3 Position, float Radius, float Intensity, float Duration)
{
	WideStringClass Netcode = StringClass::getFormattedString("j\n30\n%f\n%f\n%f\n%f\n%f\n%f\n", 
		Position.X, Position.Y, Position.Z, Radius, Intensity, Duration);

	Send_Client_Text(Netcode, TEXT_MESSAGE_PRIVATE, true, -2, PlayerID, true, true);
}

void ScreenShake_Script::Created(GameObject *obj)
{
	Duration = Get_Int_Parameter("Duration");

	if (Duration > 0)
	{
		Commands->Start_Timer(obj, this, 0.1f, 1);
		Duration--;
	}
	
//	Console_Output("Duration = %d\n", Duration); // DEBUG CRAP
}

void ScreenShake_Script::Timer_Expired(GameObject *obj, int number)
{
	if (number == 1)
	{
		Vector3 Pos = Commands->Get_Position(obj);
		Shake_Camera_Player(Get_Player_ID(obj), Pos, 1000.0f, 0.5f, 2.0f);

		if (Duration > 0)
		{
			Commands->Start_Timer(obj, this, 1.0f, 1);
			Duration--;
		}
		else
		{
			Destroy_Script();
		}
	}
}

ScriptRegistrant<ScreenShake_Script> ScreenShake_Script_Registrant("ScreenShake_Script","Duration:int");

class ScreenShakeConsoleCommand : public Plugin
{
public:
	ScreenShakeConsoleCommand()
	{
		ConsoleFunctionList.Add(new CommandSCREENSHAKE);
		Sort_Function_List();
		Verbose_Help_File();
	}
	~ScreenShakeConsoleCommand()
	{
		const DynamicVectorClass<ConsoleFunctionClass*> consoleFunctions(ConsoleFunctionList);
		TT_FOREACH(consoleFunction, consoleFunctions)
		{
			const char* name = (*consoleFunction)->Get_Name();
			if (name)
			{ 
				if (strcmp(name, "screenshake") == 0)
				{
					ConsoleFunctionList.Delete(consoleFunction);
				}
			}
		}
	}	
};

ScreenShakeConsoleCommand screenShakeConsoleCommand;

extern "C" __declspec(dllexport) Plugin* Plugin_Init()
{
	return &screenShakeConsoleCommand;
}